#import "@preview/touying:0.6.1": *
#import themes.university: *
#import "@preview/cetz:0.3.2"
#import "@preview/fletcher:0.5.5" as fletcher: node, edge
#import "@preview/numbly:0.1.0": numbly
#import "@preview/theorion:0.3.2": *
#import cosmos.clouds: *
#import "@preview/codly:1.0.0": *
#import "@preview/codly-languages:0.1.1": *

#show: codly-init.with()
#codly(languages: codly-languages)
#show: show-theorion

// cetz and fletcher bindings for touying
#let cetz-canvas = touying-reducer.with(reduce: cetz.canvas, cover: cetz.draw.hide.with(bounds: true))
#let fletcher-diagram = touying-reducer.with(reduce: fletcher.diagram, cover: fletcher.hide)

#show: university-theme.with(
  aspect-ratio: "16-9",
  // align: horizon,
  // config-common(handout: true),
  config-common(
    frozen-counters: (theorem-counter,), // freeze theorem counter for animation
    //show-notes-on-second-screen: right,
    //codly(languages: codly-languages),
    //slide-level: 3,
  ),
  config-info(
    title: [Nix/NixOS],
    subtitle: [Reproducible Builds,\ Functional Packages],
    author: [Robin Finkelmann],
    date: datetime(day: 21, month:5, year:2025, hour:18, minute:0, second: 0),
    institution: [OpenColloq Informatik \@uniwue],
    //logo: figure(image("img/nix-snowflake-colours.svg")),
  ),
)

#set heading(numbering: numbly("{1}.", default: "1.1"))

== Prepare for the Talk: <touying:hidden>

*Install Nix!*

Nix runs perfectly on any Linux Distro \
(and on macOS and WSL)

Everybody can install Nix and actively participate :)
- Visit https://determinate.systems/nix-installer/
- Follow instructions for your platform
- You can easily uninstall later

You can also use the NixOS Live ISO
- Visit https://nixos.org/download 
  - NixOS : the Linux distribution (Choose GNOME / Plasma / Minimal)

#place(top+right, image("img/nix-snowflake-colours.svg", width: 25%))

#title-slide(logo: image("img/nix-snowflake-colours.svg", width: 50%))

== Outline <touying:hidden>

#components.adaptive-columns(outline(title: none, indent: 1em, depth: 1))

= Introduction

== Motivation

Imagine...

- You use a specific version of a software (e.g. KiCAD v8)
- You need to use a different version too (e.g. a KiCAD v9 project)
- But you don't want to update your old projects yet

---

Imagine...

- You want to switch your Linux Distro's Desktop / WM
- (Or, even worse, your audio backend)

---

Imagine...

- Your laptop SSD just broke.
- You have a backup of your home dir
- Maybe even a full backup image
- But now you need to install to a smaller SSD...
- Or reinstall and reconfigure everything...

---

Imagine...

- You wish to try out a piece of software
- You install it
- You try it out
- You forget it
- It floats around forever, possibly breaking future system updates

---

Imagine...

- You have a specific problem
- You stumble upon a Git Repo that solves your problem
- It is many years old
- It uses old python packages
- It wants to install system-wide pip packages
- It has a #strike("cursed") *special* install script that copies files to /usr/bin
- The README is written for Debian, but not even Debian allows this anymore (without tweaking)

== Why Nix?

- Developing, Building, Deploying
- Reproducible, versioned Builds (write once, deploy anywhere)
- Functional Programming Language
- 1 Language for everything
- Install multiple Versions of the same Package
- Distro-Independent, even on WSL and Darwin
- Largest Package Repo of all Distros

---

#image("img/anywhere-anyplace-anytime.png")


== Why NixOS?

- Declarative Configuration
- Atomic Upgrades
- Rolling/Unstable and 6 Month Staged/Stable

== Why not Nix?
- *_TERRIBLE_* Documentation
  - Many different Formats: Nix, Nix Commands, Flakes, Nixpks, NixOS
  - Multiple different Tools for the "same" job
  - Many experimental Features are "the norm"
  - Many community projects are "the norm"
  - Aims to build everything (Eierlegende Wollmilchsau)
- Not FHS-compliant
  - Most dynamic Binaries will not work out-of-the-box
- Not 100% stable and secure
- No LTS

== Nix vs Nixpkgs vs NixOS

- Nix
  - Language / Expressions
    - Functional Language
    - Features for Building Packages (Derivation, Realisation)
    - Builtins
  - Commands
    - Nix Store
    - Nix Profiles
    - Nix Commands
    - Shells
    - Building
  - Available on all Linux Distros (and macOS)
- Nixpkgs
  - Package and Option Collection for Nix / NixOS
  - Includes Wrappers for most common Programming Languages
  - Stdenv, Lib
- NixOS
  - Linux Distro built upon Nix
  - Packages and Options from Nixpkgs
  - Manages System through Options and Modules
- Home Manager
  - Community Project
  - Manages your Dotfiles and User Environment declaratively
  - Also available on all Linux Distros (and macOS)

== Stable vs Experimental Nix

Stable Nix Commands
- `nix-<command>`, e.g. `nix-shell`
- Some are somewhat outdated
- Still occasionally used

Experimental Nix Commands
- `nix <>`, e.g. `nix shell`
- More up-to-date
- Often better
- Widely used
- No feature parity

== Ad-hoc vs Declarative Nix

- Ad-hoc refers to using Nix 'on the fly', i.e. in a shell environment
  - Ad-hoc use is optimal for experimenting
- Declarative refers to writing files that specify your actions
  - Declarative use is optimal for reuse

== Installation of Nix / Lix

Nix runs perfectly on any Linux Distro (and on macOS and WSL)

Everybody can install Nix and actively participate :)

I recommend the Nix Installer from Determinate Systems
- More deterministic, easy uninstall, Experimental Features enabled
- Visit https://determinate.systems/nix-installer/
Or if you #strike[are a bit edgy] want a faster community fork of Nix:
- Visit https://lix.systems/install/

You can also use the NixOS Live ISO (and then directly install NixOS)
- Visit https://nixos.org/download 
  - NixOS : the Linux distribution (Choose GNOME / Plasma / Minimal)

== Uninstallation of Nix

If Nix was installed with the Determinate Nix Installer (or Lix Installer), just run this script:
```bash
/nix/nix-installer uninstall
```

Otherwise, follow this guide:
- `https://nix.dev/manual/nix/2.28/installation/uninstall.html`

= Nix: The Package Manager
== nix-shell

Warning: `nix-shell` is a stable nix command.

Start a shell with GNU Hello in your environment.
```bash
nix-shell -p hello
```
You are now dropped in a Bash Shell and can run hello.
```bash
hello
```

CTRL+D quits the shell again.

---

You can also specify a command to run (instead of interactive bash):
```shell
nix-shell -p hello --run hello
```

For some more fun, I recommend this little script:
```shell
nix-shell -p lolcat cowsay --run \
"cowsay Hello, Nix! | lolcat"
```

== nix shell

Warning: `nix shell` is an experimental nix command.

Start a shell with GNU Hello in your environment.
```bash
nix shell nixpkgs#hello
```

This seems inconvenient for just using nixpkgs, but the syntax is more versatile!

Like using Repos (Flakes, more later):
```bash
nix shell github:nixos/nixpkgs#hello
hello
```

== shell.nix

Warning: `shell.nix` is a stable nix convention.

Write shells declaratively!
```nix
{ pkgs ? import <nixpkgs> {} }:
pkgs.mkShell {
  packages = [ pkgs.lolcat pkgs.cowsay ];
  inputsFrom = [];
  shellHook = ''
    echo Hello Shell!
  '';
}
```

== nix run

Warning: `nix run` is an experimental nix command.

And to run directly:
```bash
nix run github:nixos/nixpkgs#hello
```

Add branch:
```bash
nix shell github:nixos/nixpkgs/nixos-24.11#hello
```

More infos about this in later Chapter about Flakes!

== direnv

Automatically loads a shell env when entering a directory

- Install nix-direnv via Home Manager or NixOS
- Create a file called `.envrc` in the directory
- Create a `shell.nix` in the directory
-Used with default.nix, shell.nix, or a Flake (more details later)
```envrc
use nix
```

```envrc
use flake
```

https://github.com/nix-community/nix-direnv

== Install packages: nix-env

Warning: Usage not encouraged! \
Consider using Shells/Direnv, Home Manager or NixOS instead!
- Installs packages into a profile in user's home directory

```bash
nix-env -qaP fastfetch  # search for package
nix-env -iA fastfetch   # install package
nix-env -e fastfetch    # remove package
nix-env -uA fastfetch   # upgrade a package
nix-env -u              # upgrade all packages
```

https://howarddo2208.github.io/posts/02-nix-beginner-guide/

== Install packages: Home Manager

- Simply add packages to your user's environment with Home Manager
- Home Manager is also available independent of NixOS
- More about Home Manager later

```nix
  home.packages = [ pkgs.fastfetch ];
```

== Install Packages: NixOS

- Simply add packages to your system environment with NixOS
- Services must be installed using options
  - More about that later

```nix
  environment.systemPackages = [ pkgs.fastfetch ];
```

= Inner Workings

== Inspiration

Strongly inspired by:
- https://youtu.be/5D3nUU1OVx8

More info at
- https://nix.dev/tutorials/nix-language.html
- https://nixos.org/guides/nix-pills/

== Language

First things first: Nix can be interactively evaluated using:
```bash
nix repl
```

---

Comments:
```nix
# Comment
/* 
  Comment
*/
```

---

Numbers:
```nix
123     # Integer
123.4   # Float
```

---

Strings:
```nix
"abcdef"

''abc
def''

"${pkgs.echo}/bin/echo Hello world!"
```

---

List:
```nix
[ 123 "abc" ]
```

---

Attribute Set:
```nix
{ 
  name = "Nix";
  fun = 42; 
}
```

---

Functions:
```nix
# Defining
# f = ...
a: a + 1
# g = ...
{x, y ? 0}: x + y

# Calling (pretending 'f' and 'g' exists)
f 2
g {1,2}
```

---

Let in:
```nix
let
  g = {x, y ? 0}: x + y;
  f = a: g a;
in
  f 20
```

---

With (discouraged):
```nix
let
  attrs = {a = 1; b = 2; c = 3;};
in
  with attrs;
  a + b + c
```

---

Inherit:
```nix
let
  attrs = {a = 1; b = 2; c = 3;};
  inherit (attrs) a b c;
in
  a + b + c
```

---

Builtins:
e.g. `builtins.attrNames` and `buildins.functionArgs`

---

Imports:
```nix
import ./a.nix
import "./b.nix"
```

== Nix Store

- Content-addressable, immutable
- `/nix/store/<hash>-<name>-<version>/...`
- Every nix derivation and realization lives here
- Everything can be added:
  - ```bash nix store add-file```
  - ```bash nix store add-path```
- Everything is symlinked into the store
  - Results, Profiles, Binaries, Libraries, ...

== Derivations

- Declarative building instructions
- Native Nix language feature
- Produces intermediate representation `.drv`

---

File called `my-derivation.nix`
```nix
derivation {
  name = "my-program";
  system = "x86_64-linux";
  builder = "/bin/bash";
  src = ./main.c;
  args = [ "-c" ''
    /usr/bin/clang $src
  '']
}
```

---

```bash
nix-instantiate my-derivation.nix
nix derivation show <store-path>

nix-store --realize <store-path>
```

---

File called `my-derivation.nix`
```nix
derivation {
  name = "my-program";
  system = "x86_64-linux";
  builder = "/bin/bash";
  src = ./main.c;
  args = [ "-c" ''
    /usr/bin/clang $src -o $out
  '']
}
```

--- 

```bash
nix-build my-derivation.nix
```

Depending on environment, either succeeds or fails.

Heavily impure!

---

#figure(
  image("img/bootstrap.png", height: 70%), 
  caption: [Proper bootstrapping of all build tools into Nix],
)

This is where Nixpkgs and Stdenv come into play!

---

```nix
{ 
  pkgs ? import (fetchTarball "https://github.com/NixOS/nixpkgs/archive/06278c77b5d162e62df170fec307e83f1812d94b.tar.gz") {}
}:
derivation {
  name = "my-program";
  system = "x86_64-linux";
  builder = "${pkgs.bash}/bin/bash";
  src = ./main.c;
  args = [ "-c" ''
    ${pkgs.clang}/bin/clang $src -o $out
  '']
}
```

---

```nix
{ 
  pkgs ? import (fetchTarball "https://github.com/NixOS/nixpkgs/archive/06278c77b5d162e62df170fec307e83f1812d94b.tar.gz") {}
}:
pkgs.stdenv.mkDerivation {
  name = "my-program";
  system = "x86_64-linux";
  nativeBuildInputs = [];   # Build-Time
  buildInputs = [];         # Runtime
  dontUnpack = true;
  buildPhase = ''
    clang $src -o my-program
  '';
  installPhase = ''
    mkdir -p $out/bin
    cp my-program $out/bin$
  '';
}
```

== Nixpkgs

#figure(
  image("img/repos.png", height: 70%), 
  caption: [Linux Distro's Repo's Package Counts],
)

- https://repology.org/repositories/graphs

---

- https://github.com/NixOS/nixpkgs

== Channels

Warning: Channels are a stable nix feature.

- Provide an atomic version of nixpkgs

For Nix:
```bash
nix-channel --list
nix-channel --add https://nixos.org/channels/nixpkgs-unstable nixpkgs
nix-channel --update
```

For NixOS
```bash
nix-channel --list
nix-channel --add https://nixos.org/channels/nixos-unstable nixos
nix-channel --add https://nixos.org/channels/nixos-25.05 nixos
nix-channel --update
```

== NARs

Nix Archives

https://nix.dev/manual/nix/2.22/protocols/nix-archive

== Profiles, GC

Profiles: Atomic collection of symlinks into the Nix Store 

#figure(
  image("img/profiles.png", width: 80%),
  caption: "https://nix.dev/manual/nix/2.22/protocols/nix-archive",
)

---

- `nix-env` manipulates user-profiles
- NixOS manipulates system profiles

```bash
ls -la /nix/var/nix/profiles/
```

---

Store paths can be marked as Garbage Collection roots.
- GC Roots will never be deleted
- Dependencies will never be deleted

```bash
nix-env --delete-generations old
nix-store --gc
nix-collect-garbage -d
nix-collect-garbage --delete-older-than 30d

nix profile ...
nix gc ...
```

= Nix: The Build System

== nix-build (ad-hoc)

Warning: `nix-build` is a stable nix command.

```shell
nix-build -E "with import <nixpkgs> { }; hello"
./result/bin/hello
rm result
```

== nix-build (default.nix)

Warning: `default.nix` is a stable nix convention.

Create a file named `default.nix` with the following content:
```nix
let
  pkgs = import <nixpkgs> {};
in
pkgs.hello
```
Build this expression using `nix-build`. A result symlink will appear.
```shell
nix-build
./result/bin/hello
rm result
```

== nix vs nix flakes

Both `default.nix` and `shell.nix` are the stable Nix way of doing things. The experimental successor is Flakes.

I try to use flakes whenever possible, only using ad-hoc, `default.nix` and `shell.nix` for quick'n'dirty usecases.

== Flakes

Flakes! Not scary at all!

However, a lot of confusion what flakes actually are

First, take a look at a flake:

```bash
nix flake init
cat flake.nix
nix flake show
cat flake.lock
nix flake update
```

---

```nix
{
  description = "A very basic flake";
  inputs = {
    nixpkgs.url = "github:nixos/nixpkgs?ref=nixos-unstable";
  };
  outputs = { self, nixpkgs }: {
    packages.x86_64-linux.hello = nixpkgs.legacyPackages.x86_64-linux.hello;
    packages.x86_64-linux.default = self.packages.x86_64-linux.hello;
  };
}
```

---

So, a flake is only another way to write nix expressions
- Specify inputs
- Specify outputs
- Less implied context like default.nix, shell.nix or nixpkgs

---

Ease of use:
- Combine all possible outputs into one file `flake.nix` 
  - We know `shell.nix` and `default.nix`, but also NixOS Configs etc.
- Flakes can easily be used as inputs for other flakes 
  - Typically, `nixpkgs` is used as a flake input

Reproducability out of the box:
- Locks inputs in a `flake.lock` file (just like Rust's `cargo.lock`)
  - In most cases a Link (e.g. Git Revision) and a Hash
- Exactly specify which architecture a package is for
  - i.e. `x86_64-linux`

But also, added complexity (no sane person can write flakes by memory)

---

Now, look at a comlpete flake template

```nix
{
  description = "A very basic flake";

  inputs = {
    nixpkgs.url = "github:nixos/nixpkgs?ref=nixos-unstable";
  };

  outputs = { self, ... }@inputs:
  {
    # Executed by `nix flake check`
    checks."<system>"."<name>" = derivation;
    # Executed by `nix build .#<name>`
    packages."<system>"."<name>" = derivation;
    # Executed by `nix build .`
    packages."<system>".default = derivation;
    # Executed by `nix run .#<name>`
    apps."<system>"."<name>" = {
      type = "app";
      program = "<store-path>";
    };
    # Executed by `nix run . -- <args?>`
    apps."<system>".default = { type = "app"; program = "..."; };

    # Formatter (alejandra, nixfmt or nixpkgs-fmt)
    formatter."<system>" = derivation;
    # Used for nixpkgs packages, also accessible via `nix build .#<name>`
    legacyPackages."<system>"."<name>" = derivation;
    # Overlay, consumed by other flakes
    overlays."<name>" = final: prev: { };
    # Default overlay
    overlays.default = final: prev: { };
    # Nixos module, consumed by other flakes
    nixosModules."<name>" = { config, ... }: { options = {}; config = {}; };
    # Default module
    nixosModules.default = { config, ... }: { options = {}; config = {}; };
    # Used with `nixos-rebuild switch --flake .#<hostname>`
    # nixosConfigurations."<hostname>".config.system.build.toplevel must be a derivation
    nixosConfigurations."<hostname>" = {};
    # Used by `nix develop .#<name>`
    devShells."<system>"."<name>" = derivation;
    # Used by `nix develop`
    devShells."<system>".default = derivation;
    # Hydra build jobs
    hydraJobs."<attr>"."<system>" = derivation;
    # Used by `nix flake init -t <flake>#<name>`
    templates."<name>" = {
      path = "<store-path>";
      description = "template description goes here?";
    };
    # Used by `nix flake init -t <flake>`
    templates.default = { path = "<store-path>"; description = ""; };
  }
}
```

https://wiki.nixos.org/wiki/Flakes
---

Some other important tools related to flakes:
- flake-compat: Interface between `default.nix`, `shell.nix` and `flake.nix`
  - https://github.com/edolstra/flake-compat
- flake-utils: easily use multiple systems
  - https://github.com/numtide/flake-utils

---

Let's have a look at more Flakes!

- Behold, my NixOS Config!

- Nixpkgs

- 4rth wall break

== nix develop

Enters a development environment for a given package.
Useful for debugging the build process.

```bash
nix develop nixpkgs#hello
unpackPhase
cd <name>
configurePhase
mkdir build && cd build
buildPhase
checkPhase
installPhase
installCheckPhase
../outputs/out/bin/hello
```

CTRL+D quits again.

== Shells / nix shell

Now you better understand how Nix works, how does this work:

```bash
nix shell nixpkgs#hello
```

---

Shells can be put inside a Flake.

```bash
nix flake init -t "github:determinatesystems/zero-to-nix#cpp-dev"
```

```bash
nix flake init -t "github:determinatesystems/zero-to-nix#rust-dev"
```

```bash
nix shell
```


== Packages / nix build 

Packages can be put inside a Flake.

```bash
nix flake init -t "github:determinatesystems/zero-to-nix#cpp-pkg"
```

```bash
nix flake init -t "github:determinatesystems/zero-to-nix#rust-pkg"
```

```bash
nix build
```


== A "better" Nix Command

`nix-output-monitor` makes the building process prettier
- Drop-in wrapper for
  - `nix-build` and `nix build`
  - `nix-shell` and `nix shell`
  - `nix develop`
- Just replace `nix` with `nom`

Test it out like this:
```bash
nix shell nixpkgs#nix-output-monitor
nom shell nixpkgs#linux --no-substitute
```

= Other Stuff

== NixOS

- https://nixos.org/manual/nixos/stable/

=== configuration.nix

configuration.nix

=== hardware-configuration.nix

hardware-configuration.nix

=== The Module System

modules

=== Options

Options

=== nh

- https://github.com/nix-community/nh

=== nix-tree

- https://github.com/utdemir/nix-tree

== Home Manager

Home Manager declaratively manages your home's dotfiles.

```nix
  programs.git = {
    enable = true;
    userName  = "my_git_username";
    userEmail = "my_git_username@gmail.com";
  };
  programs.direnv = {
    enable = true;
    nix-direnv.enable = true;
  };
  programs.fish.enable = true;
```

---

```nix
  programs.git = {
    enable = true;
    userName  = "my_git_username";
    userEmail = "my_git_username@gmail.com";
  };
  programs.direnv = {
    enable = true;
    nix-direnv.enable = true;
  };
```

== Fix dynamically linked Binaries

For running most binaries, add `pkgs.autoPatchelfHook` to your env, either ad-hoc or in the nativeBuildInputs of a package.
```bash 
nix shell nixpkgs#autoPatchelfHook
```

Or write a FHS env.
Or do whatever this is:

#figure(
  image("img/fhs.png", ), 
  caption: [Official FAQ from https://nix.dev/guides/faq],
)

== FHS

Nixpkgs provides a `pkgs.buildFHSEnv` function, calling `.env` on it drops you in its shell.

```nix
{ pkgs ? import <nixpkgs> {} }:
  (pkgs.buildFHSEnv {
    name = "buildroot-fhs-env";
    multiPkgs = pkgs: (with pkgs; [ hello ]);
    runScript = "fish";
  }).env
```

https://nixos.org/manual/nixpkgs/stable/#sec-fhs-environments

== Cross Compilation

Cross compilation can be achieved by using `pkgsCross.<architecture>` instead of `pkgs`, e.g by executing

```bash
nix build nixpkgs#pkgsCross.riscv64.hello
```

== CUDA

CUDA

== Binary Caching, Cachix

Binary Caching, Cachix

== Sharing Nix Stores

Sharing Nix Stores

== Distributed Builds

Distributed Builds

https://nix.dev/tutorials/nixos/distributed-builds-setup.html

== Hydra

Hydra 

https://github.com/NixOS/hydra
https://wiki.nixos.org/wiki/Hydra
https://hydra.nixos.org/

== Darwin

Darwin

https://github.com/nix-darwin/nix-darwin

== Disko

#figure(
  image("img/disko.png", height: 50%,), 
  caption: [Official `Disko` Logo],
)

`Disko` enables declarative Disk Partitioning for NixOS\
- `https://github.com/nix-community/disko`

== SOPS / Age-Nix

- https://github.com/ryantm/agenix (recommended)
- https://github.com/Mic92/sops-nix (also possible)

#show: appendix

= Appendix

== Important Websites

Nix: Package Management
- `https://search.nixos.org/`
  - Official Package and Option Index
- `https://github.com/nixos/nixpks`
  - Repo for Nixpkgs

---

Nix: Development
- `https://nix.dev/`
  - Good Reference for using Nix productively, especially for Devs
- `https://noogle.dev/`
  - Nix Function Search, Function of the day
- `https://zero-to-nix.com/`
  - Very good tutorials!
- `https://nixos.org/manual/nixpkgs/stable/`
  - Official Manual for Nixpkgs

---

NixOS
- `https://wiki.nixos.org/` 
  - Official Wiki, good for NixOS users
  - Warning: The older Community Wiki `https://nixos.wiki` is still online and often pops up when googling!
- `https://nixos.org/manual/nixos/stable/`
  - Official Manual for NixOS
- `https://github.com/nix-community/nixos-hardware`
  - Repo for NixOS modules for specific hardware
- `https://www.youtube.com/@vimjoyer`
  - Good, quick tutorials

---

Just great:
- `https://github.com/nix-community/awesome-nix`
  - List of many community projects
- `https://nixos.org/guides/nix-pills/`
  - Explaining Nix from the ground up (tho somewhat dated)
- `https://edolstra.github.io/pubs/phd-thesis.pdf`
  - The PHD of Eelco Dolstra about developing Nix

== Other stuff

#figure(
  image("img/amazon.png", height: 90%), 
)

---

`https://guix.gnu.org/`
- Guile Scheme as Language
- No non-FOSS packages
- Hurd Kernel sometime?

https://luj.fr/blog/how-nixos-could-have-detected-xz.html

https://guix.gnu.org/en/blog/2023/the-full-source-bootstrap-building-from-source-all-the-way-down/
