import os
import urllib.request
import tarfile
import pickle
import numpy as np
from typing import Tuple

def load_cifar10_batch(file: str) -> Tuple[np.ndarray, np.ndarray]:
    """Load a batch of the CIFAR-10 dataset.

    Args:
        file (str): The file to load the data from.

    Returns:
        Tuple[np.ndarray, np.ndarray]: The data and labels of the batch.
    """
    with open(file, 'rb') as fo:
        dict = pickle.load(fo, encoding='bytes')
    return dict[b'data'], np.array(dict[b'labels'])

def download_cifar10() -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray]:
    """Download and load the CIFAR-10 dataset.

    Returns:
        Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray]: The training data, training labels, test data, and test labels.
    """

    url = "https://www.cs.toronto.edu/~kriz/cifar-10-python.tar.gz"
    file_name = "cifar-10-python.tar.gz"
    data_dir = "cifar-10-batches-py"

    if not os.path.exists(data_dir):
        if not os.path.isfile(file_name):
            urllib.request.urlretrieve(url, file_name)

        with tarfile.open(file_name, "r:gz") as tar:
            tar.extractall()

    x_train = []
    y_train = []
    for i in range(1, 6):
        data, labels = load_cifar10_batch(os.path.join(data_dir, f'data_batch_{i}'))
        x_train.append(data)
        y_train.append(labels)
    x_train = np.concatenate(x_train)
    x_train = x_train.reshape(-1, 3, 32, 32).transpose([0, 2, 3, 1])
    y_train = np.concatenate(y_train)

    x_test, y_test = load_cifar10_batch(os.path.join(data_dir, 'test_batch'))
    x_test = x_test.reshape(-1, 3, 32, 32).transpose([0, 2, 3, 1])

    return x_train, y_train, x_test, y_test