#!/usr/bin/env python
# coding: utf-8

# # Dynamic Programming: Levenshtein Distance
# 
# In this programming exercise, you are going to implement the Levenshtein distance in a simple Python function with dynamic programming.
# 
# 
# Memoization in Python typically relies on the use of 
# functools.cache: https://docs.python.org/3/library/functools.html#functools.cache
# or
# functools.lru_cache: https://docs.python.org/3/library/functools.html#functools.lru_cache
# 
# You can find a an accessible introduction to decorators here:
#     https://realpython.com/primer-on-python-decorators/
# 
# You may quite likely find that memoization for levenshtein distance is implemented
# better with a little custom decorator.

# The below `call_counter` decorator demonstrates the utiltity of memoization in recursive programming on the case of computing the fibonacci sequence for `n`.

# In[1]:


def call_counter(func):
    """Simple decorator to keep track of number of function calls."""
    def helper(*args, **kwargs):
        helper.calls += 1
        return func(*args, **kwargs)
    helper.calls = 0
    helper.__name__= func.__name__
    return helper


# To showcase memoization, we will implement the Fibonacci sequence recursively with (`fibonacci_cached`) and without memoization (`fibonacci`) and track the number of function calls.

# In[2]:


# the below syntax means that `call_counter` decorates `fibonacci`
@call_counter
def fibonacci(n):
	if n == 0:
		return 0

	# Check if n is 1,2
	# it will return 1
	elif n == 1 or n == 2:
		return 1

	else:
		return fibonacci(n-1) + fibonacci(n-2)

print(f"Result: {fibonacci(9)}")
print(f"Number of function calls: {fibonacci.calls}")  # 67 calls


# In[3]:


# we now also lever the `cache` decorator from the Python standard library 
from functools import cache
@call_counter
@cache
def fibonacci_cached(n):
	if n == 0:
		return 0

	# Check if n is 1,2
	# it will return 1
	elif n == 1 or n == 2:
		return 1

	else:
		return fibonacci_cached(n-1) + fibonacci_cached(n-2)

print(f"Result: {fibonacci_cached(9)}")
print(f"Number of function calls: {fibonacci_cached.calls}")  # 67 calls


# In[7]:


def levenshtein_distance(a: str, b: str) -> int:
    """
    Important: your solution must use memoization!

    Memoization in Python typically relies on the use of 
    functools.cache: https://docs.python.org/3/library/functools.html#functools.cache
    or
    functools.lru_cache: https://docs.python.org/3/library/functools.html#functools.lru_cache

    You can find a an accessible introduction to decorators here:
        https://realpython.com/primer-on-python-decorators/

    An example of a decorator is the `call_counter` function that decorates
    the `fibonacci_cached` function.

    You may quite likely find that memoization for levenshtein distance is implemented better with a little custom decorator.
    """
    pass


import unittest

class LevenshteinDistanceTest(unittest.TestCase):
    def test_levenshtein_distance(self):
        self.assertEqual(0, levenshtein_distance("", ""))
        self.assertEqual(0, levenshtein_distance("aa", "aa"))
        self.assertEqual(1, levenshtein_distance("aa", "aaa"))
        self.assertEqual(1, levenshtein_distance("abc", "aec"))
        self.assertEqual(
            14,
            levenshtein_distance(
                "We compute the levenshtein distance", "We measure the edit distance"
            ),
        )


unittest.main(argv=[""], verbosity=2, exit=False)


# **Bonus question:** can you compare and explain the efficiency of your Levenshtein distance implementation with memoization to a variant without memoization? Good and valid answers will receive half a bonus point!
