#!/usr/bin/env python
# coding: utf-8

# # Binary Search and AVL trees
#
# In this exercise we will implement commonly used methods of binary search trees (AVL-trees).
#
# The implementation of `insert` and `delete` is optional. As an incentive, _should_ you decide to implement it to hone your understanding of AVL-trees, a correct implementation will be automatically awarded with full points (i.e. 2) for this exercise. Why? To appropriately implement insert and delete, you will require some of the other functions and the pen-and-paper exercises the become a subset of what you are required for these two functions.

# In[1]:


from typing import Optional, Union, List, Tuple

# set for _settattr_ checking
NODE_ATTRS = {"left", "right", "value", "parent"}
# we are executing utils.py to "import" the functionality to print the nodes without having to setup a proper module
# IMPORTANT!: make sure utils.py is in the same folder as your ipython notebook!


def _build_tree_string(
    root,
    curr_index: int,
    include_index: bool = False,
    delimiter: str = "-",
) -> Tuple[List[str], int, int, int]:
    """Utility function to print a string representation of the tree spanned by `Node`."""
    if root is None:
        return [], 0, 0, 0

    line1 = []
    line2 = []
    if include_index:
        node_repr = "{}{}{}".format(curr_index, delimiter, root.value)
    else:
        node_repr = str(root.value)

    new_root_width = gap_size = len(node_repr)

    # Get the left and right sub-boxes, their widths, and root repr positions
    l_box, l_box_width, l_root_start, l_root_end = _build_tree_string(
        root.left, 2 * curr_index + 1, include_index, delimiter
    )
    r_box, r_box_width, r_root_start, r_root_end = _build_tree_string(
        root.right, 2 * curr_index + 2, include_index, delimiter
    )

    # Draw the branch connecting the current root node to the left sub-box
    # Pad the line with whitespaces where necessary
    if l_box_width > 0:
        l_root = (l_root_start + l_root_end) // 2 + 1
        line1.append(" " * (l_root + 1))
        line1.append("_" * (l_box_width - l_root))
        line2.append(" " * l_root + "/")
        line2.append(" " * (l_box_width - l_root))
        new_root_start = l_box_width + 1
        gap_size += 1
    else:
        new_root_start = 0

    # Draw the representation of the current root node
    line1.append(node_repr)
    line2.append(" " * new_root_width)

    # Draw the branch connecting the current root node to the right sub-box
    # Pad the line with whitespaces where necessary
    if r_box_width > 0:
        r_root = (r_root_start + r_root_end) // 2
        line1.append("_" * r_root)
        line1.append(" " * (r_box_width - r_root + 1))
        line2.append(" " * r_root + "\\")
        line2.append(" " * (r_box_width - r_root))
        gap_size += 1
    new_root_end = new_root_start + new_root_width - 1

    # Combine the left and right sub-boxes with the branches drawn above
    gap = " " * gap_size
    new_box = ["".join(line1), "".join(line2)]
    for i in range(max(len(l_box), len(r_box))):
        l_line = l_box[i] if i < len(l_box) else " " * l_box_width
        r_line = r_box[i] if i < len(r_box) else " " * r_box_width
        new_box.append(l_line + gap + r_line)

    # Return the new box, its width and its root repr positions
    return new_box, len(new_box[0]), new_root_start, new_root_end


class Node:
    """A simple binary tree node.


    Params:
        value: value associated with `Node`
        left: left child `Node` of node
        right: right child `Node` of node
        parent: pointer to parent node to be able to walk up the tree
    Methods:
        __setattr__:
            Allows you to set the `value`, `left` or `right` attributes of the `Node`

            node = Node(3)
            node.value = 5

        __str__:
            You can print a clean representation of the tree for debugging purposes with `print(node)`.

            >> root = Node(10)
            >> root.left = Node(1)
            >> root.right = Node(3)
            >> root.right.right = Node(30)
            >> print(root)
            >>   10
            >>  /  \
            >> 1    3
            >>       \
            >>        30
    """

    def __init__(
        self,
        value: int,
        left: Optional["Node"] = None,
        right: Optional["Node"] = None,
        parent: Optional["Node"] = None,
    ) -> None:
        """Initialize binary tree node.

        Args:
            value: value associated with `Node`
            left: left child `Node` of node
            right: right child `Node` of node
            parent: pointer to parent `Node`

        Returns:
            None: instantiated node
        """
        self.value = value
        self.left = left
        self.right = right
        self.parent = parent

    def __setattr__(self, attr: str, obj: Union[int, "Node"]) -> None:
        """``__setattr__`` with type checking."""
        assert (
            attr in NODE_ATTRS
        ), f"{attr} is not a valid Node attribute ({NODE_ATTRS})"
        object.__setattr__(self, attr, obj)

    def __str__(self) -> str:
        lines = _build_tree_string(self, 0, False, "-")[0]
        return "\n" + "\n".join((line.rstrip() for line in lines))


# We provide a utility function to print trees into a string representation that may help you debugging your implementations.
# Note: The below cell only works if `utils.py` is in the same folder as the Ipython Notebook.

# In[2]:


node = Node(3)
node.left = Node(4)
node.right = Node(5)
print(node)


# Get the minimum of the binary search sub-tree spanned by `Node` root.

# In[3]:


def get_min(root: Node) -> int:
    """Get the minimum of the binary search tree spanned by `root`.

    Args:
        root: root `Node` of the tree.

    Returns:
        int: the minimum value of the binary search tree

    Implementation:
        - Lever binary search tree property and recursively traverse left child until no child found
        - Return value of leaf node
    """
    left_child = root.left
    minimum = root.value
    while left_child is not None:
        minimum = left_child.value
        left_child = left_child.left
    return minimum


# Get the maximum of the binary search sub-tree spanned by `Node` root.

# In[4]:


def get_max(root: Node) -> int:
    """Get the maximum of the binary search tree spanned by `root`.

    Args:
        root: root `Node` of the tree.

    Returns:
        int: the maximum value of the binary search tree

    Implementation:
        - Lever binary search tree property and recursively traverse right child until no child found
        - Return value of leaf node
    """
    right_child = root.right
    maximum = root.value
    while right_child is not None:
        maximum = right_child.value
        right_child = right_child.right
    return maximum


# Search for the value in the binary sub-tree spanned by `Node` root.

# In[5]:


def search(root: Optional[Node], value: int) -> Optional[Node]:
    """Returns the `Node` of `value` if found in tree.

    Assume:
        - root spans binary search tree
        - if parent and child have the same value, return child

    Args:
        root: root `Node` of the tree.
        value: Integer to `search` for in the tree

    Returns:
        Optional[Node]: return the `Node` of `value` if `value` found in binary search tree of `root`

    Implementation:
        - Recursively search both left (right) downwards if value is smaller (larger) equal to value of current root node

    """
    if root is None:
        return
    root_match = root.value == value
    # traverse left child if smaller or equal to right hand-side of value
    left_search = search(root.left, value) if root.value >= value else None
    # traverse right child if larger or equal to right hand-side of value
    right_search = search(root.right, value) if root.value <= value else None

    # iteratively check what branches have returned matches
    # 1) Set return value to root if root matches value
    ret = root if root_match else None
    # 2) Set to left if it matches left value, else keep original value
    ret = ret if not left_search else left_search
    # 3) Prefer right match if exists, otherwise keep prior value
    ret = ret if not right_search else right_search
    return ret


# ### Balance in binary search trees (AVL trees)
#
# The constraint is generally applied recursively to every subtree. That is, the tree is only balanced if:
#
#     The left and right subtrees' heights differ by at most one, AND
#     The left subtree is balanced, AND
#     The right subtree is balanced
#
# According to this, the next tree is balanced:
#
# <pre>
#       A
#     /   \
#    B     C
#   /     / \
#  D     E   F
#       /
#      G
# </pre>
#
# The next one is not balanced because the subtrees of C differ by 2 in their height:
#
# <pre>
#      A
#    /   \
#   B     C   <-- difference = 2
#  /     /
# D     E
#      /
#     G
# </pre>
#
# That said, the specific constraint of the first point depends on the type of tree. The one listed above is the typical for AVL trees.\
# (Creds to this nice outline: https://stackoverflow.com/a/14712245)
#
# Test whether the sub-tree spanned by `Node` `root` is balanced.
#
# Note: should you implement insertion and deletion, think about how you will require the height in other functions and potentially implement another helper function that implements the core logic of `is_balanced` working primarily with tree heights)

# In[6]:


def height(node: Optional[Node]) -> int:
    """Return the height of the sub-tree spanned by node.


    Args:
        node: Node or None

    Returns:
        int: height of sub-tree spanned by Node

    Implementation:
        - If `Node` has no children or is `None` `return 0`
        - Else `return 1 + max(height(left child), height(right cild))`
    """
    if node is None:
        return 0
    # while can go left or right down, increment with 1
    h = 0
    left_child = node.left
    right_child = node.right
    if left_child is not None or right_child is not None:
        h += 1
    left_height = height(left_child)
    right_height = height(right_child)
    return h + max(left_height, right_height)


def is_balanced(root: Optional[Node]) -> bool:
    """Returns whether the tree spanned by the sub-tree of `root` is balanced.

    As per the lecture, a tree is balanced if the balance factor is valid.
    (cf. slide 13)

    The balance factor is defined as follows
        bf(node) = height(node.right) - height(node.left)
    and must be in {-1, 0, 1}

    Args:
        root: the root node of the binary tree.

    Returns:
        bool: whether or not the tree is balanced.

    Preface:
        The constraint is generally applied recursively to every subtree. That is the tree is only balanced if:
        
            The left and right subtrees' heights differ by at most one AND
            The left subtree is balanced AND
            The right subtree is balanced
        
        According to this the next tree is balanced:
            <pre>
                  A
                /   \\  # just escaping here with double \\
               B     C  
              /     / \\ 
             D     E   F  
                  /  
                 G  
            </pre>
        
        The next one is not balanced because the subtrees of C differ by 2 in their height:
        
                 A
               /   \\
              B     C   <-- difference = 2
             /     /
            D     E  
                 /  
                G  
        

    Recursive Implementation:
        - A key aspect of the recursion is that the height of the tree spanned by node is 1 + height of height of its children
        For each node in the tree, check that
        - The sub-tree spanned by the node is balanced, that is the height of left-hand and right-hand tree differ by at most 1
        - If imbalanced return False (i.e. exit condition in recursion)
        - Else check that both left child AND right child are balanced tree's; the AND operator ensures that if any sub-tree is imbalanced we will return False
    """
    if root is None:
        return True
    left_child = root.left
    right_child = root.right
    # IMPORTANT: height of the tree spanned by node is 1 + height of height of its children
    left_height = height(left_child) + 1 if left_child is not None else 0
    right_height = height(right_child) + 1 if right_child is not None else 0
    # if it's false for parent, don't have to check for children
    if not abs(left_height - right_height) < 2:
        return False
    return is_balanced(left_child) and is_balanced(right_child)


# Test whether the sub-tree spanned by `Node` `root` is balanced.

# In[7]:


def is_binary_search_tree(
    root: Optional[Node], parents: Optional[list[Node]] = None
) -> bool:
    """Check if the binary tree is a BST (binary search tree).

    Args:
        root: Root node of the binary tree.
    Returns:
        bool: `True` if the binary tree is a binary search tree, `False` otherwise.

    Recursive Implementation:
        - Lever binary search tree propery
        - A sub-tree spanned by root is a binary search tree iff
            - The property holds wrt its children
            - The sub-trees spanned by left and right child, respectively, are BST
    """
    if root is None:
        return True

    # INFO!: the below is not required for the test to pass
    #        we most likely won't make this a requirement
    # if root is not the root node if the tree
    # we have to check that binary search tree property of current root is
    # fulfilled with respect to all of its parents
    # for that we collect parents in a list (descendingly, last parent is immediate parent)
    # we then check on what side of the current parent level node is on
    # this we check with the relationship between current_parent and its child
    if parents is not None:
        i = len(parents) - 1
        current_child = root
        while i >= 0:
            current_parent = parents[i]
            is_right = True if current_parent.right is current_child else False
            if is_right:
                if not (root.value >= current_parent.value):
                    return False
            else:
                if not (root.value <= current_parent.value):
                    return False
            current_child = current_parent
            i -= 1

    left_child = root.left
    right_child = root.right
    left_valid = left_child.value <= root.value if left_child is not None else True
    right_valid = right_child.value >= root.value if right_child is not None else True
    # early exit condition of recursion
    if not (left_valid and right_valid):
        return False
    # if we didn't exit, continue for children
    # list(parents) makes a shallow copy of list (a copy of pointers), see https://stackoverflow.com/a/2612815
    # alternatively import copy; copy.copy(parents)
    parents = list(parents) if isinstance(parents, list) else []
    parents.append(root)
    return is_binary_search_tree(left_child, parents) and is_binary_search_tree(
        right_child, parents
    )


# Implement insertion for `AVL-trees`. The implementation is analogous to the modular pseudo-code of the lecture.

# In[8]:


def _append(node: Node, value: int) -> Node:
    # we always want to append to the right even in case of equality
    is_larger_eq = value >= node.value
    if not is_larger_eq:
        if node.left is None:
            new_node = Node(value)
            new_node.parent = node
            node.left = new_node
        else:
            new_node = _append(node.left, value)
    else:
        if node.right is None:
            new_node = Node(value)
            new_node.parent = node
            node.right = new_node
        else:
            new_node = _append(node.right, value)
    return new_node


def left_rotation(node: Node):
    """Perform a left rotation to rebalance the binary search tree.

    Args:
        node: node in which spans a sub-tree that is imbalanced
    """
    # imagine node is
    #   3
    #    \
    #     5
    #      \
    #       6
    assert node.right is not None and node.right.right is not None
    # flip values
    node.right.value, node.value = node.value, node.right.value
    # after flipping
    #   5
    #    \
    #     3
    #      \
    #       6
    # 6 (node.right.right) is the new right of 5 (now node)
    new_right = node.right.right
    new_right.parent = node
    node.right.right = None  # IMPORTANT: cut connection appropriately!

    # 3 (node.right) is the new left of 5 (now node)
    new_left = node.right
    # it's important to consider how the corresponding sub-trees move
    # what previously was left of 5 is now right of 3
    # what was left of 3 __remains__ left of 3 in new position of 3
    new_left.right, new_left.left = node.left, new_left.right

    node.left = new_left
    node.right = new_right


def right_rotation(node: Node):
    """Perform a right rotation to rebalance the binary search tree.

    Args:
        node: node in which spans a sub-tree that is imbalanced
    """
    # imagine node is

    #      6
    #     /
    #    4
    #   /
    #  3
    assert node.left is not None and node.left.left is not None
    # flip values
    node.left.value, node.value = node.value, node.left.value
    #      4
    #     /
    #    6
    #   /
    #  3
    # 3 (node.left.left) is the new left of 4 (now node)
    new_left = node.left.left
    new_left.parent = node
    node.left.left = None  # IMPORTANT: cut connection appropriately!

    # it's important to consider how the corresponding sub-trees move
    # what previously was left of 6 is now right of 4
    # what was right of 4 __remains__ right of 4 in new position of 6
    new_right = node.left
    new_right.left, new_right.right = node.right, new_right.left

    node.left = new_left
    node.right = new_right


def right_left_rotation(node: Node):
    assert node.right is not None and node.right.left is not None
    # flip values
    node.value, node.right.left.value = node.right.left.value, node.value

    # fix bottom up
    new_left = node.right.left
    if node.right.left is not None:
        node.right.left.parent = node.right

    # what's now new node.right.left is what intermittently is at new_left.right
    node.right.left = new_left.right
    # now fix new_left.right
    new_left.right, new_left.left = new_left.left, node.left
    if new_left.right is not None:
        new_left.right.parent = new_left
    if new_left.left is not None:
        new_left.left.parent = new_left
    node.left = new_left
    new_left.parent = node


def left_right_rotation(node: Node):
    assert node.left is not None and node.left.right is not None
    # flip values
    node.value, node.left.right.value = node.left.right.value, node.value

    # fix bottom up
    new_right = node.left.right
    # what's now new node.right.left is what intermittently is at new_left.right
    node.left.right = new_right.left
    if node.left.right is not None:
        node.left.right.parent = node.left
    # now fix new_left.right
    new_right.left, new_right.right = new_right.right, node.right
    if new_right.left is not None:
        new_right.left.parent = new_right
    if new_right.right is not None:
        new_right.right.parent = new_right
    node.right = new_right
    new_right.parent = node


def _rebalance(root: Node):
    # determine type of imbalance
    left_node = root.left
    right_node = root.right
    left_node_height = height(left_node) if left_node is not None else -1
    right_node_height = height(right_node) if right_node is not None else -1
    # left imbalance
    if left_node_height > right_node_height:
        # check left left
        left_node_left_height = (
            height(left_node.left) if left_node.left is not None else -1
        )
        left_node_right_height = (
            height(left_node.right) if left_node.right is not None else -1
        )
        if left_node_left_height > left_node_right_height:
            right_rotation(root)
        else:
            left_right_rotation(root)
    else:
        right_node_left_height = (
            height(right_node.left) if right_node.left is not None else -1
        )
        right_node_right_height = (
            height(right_node.right) if right_node.right is not None else -1
        )
        if right_node_right_height > right_node_left_height:
            left_rotation(root)
        else:
            right_left_rotation(root)


def insert(root: Node, value: int) -> None:
    """Insert `value` into `root`, maintaining AVL-tree properties.

    This requires you to implement the appropriate rotations modularly in separate functions.

    Note:
        - You will have to create a `Node` from `value`

    Args:
        root: Root node of the binary tree.
    """
    new_node = _append(root, value)
    current_node = new_node
    parent = current_node.parent
    while parent is not None:
        if is_balanced(parent):
            current_node = current_node.parent
            parent = current_node.parent
        else:
            _rebalance(parent)
            break
    assert is_balanced(parent)


# Implement deletion for `AVL-trees`. The implementation is analogous to the modular pseudo-code of the lecture.

# In[9]:


def remove_or_replace(node: Node, replacement: Optional[Node] = None) -> None:
    """`node` is removed from or replaced with `replacement` in its parent."""
    parent = node.parent
    if parent is not None:
        if parent.left is node:
            parent.left = replacement
        else:
            parent.right = replacement
        if replacement is not None:
            replacement.parent = parent


def delete(root: Optional[Node], value: int, kind="successor") -> None:
    """Delete value and maintain AVL-tree properties.

    Requires:
        - Identifying the node with `value` in the tree spanned by `Node` root
        - Implementing the appropriate rotations (cf. insert)
    """
    # find node
    node = search(root, value)
    # node to start checking tree balance from, may change depending on deletion scenario
    check_balance_node = node
    # node not found, do nothing
    if node is None:
        return None

    # test cases
    left_child = node.left
    right_child = node.right

    # case 1: delete node with no children
    if left_child is None and right_child is None:
        parent = node.parent
        if parent is not None:
            remove_or_replace(node)
            check_balance_node = parent
        else:
            root = None
            check_balance_node = root
    # case 2: only 1 child
    # ^ is xor
    # (left is None) -> expands to boolean; parentheses required for appropriate evaluation
    elif (left_child is None) ^ (right_child is None):
        child = left_child if right_child is None else right_child
        assert child is not None  # satisfy linter
        # exchange values
        child.value, node.value = node.value, child.value
        # set pointers of children
        node.right = child.right
        node.left = child.left
        # set parents of children appropriately
        if node.left is not None:
            node.left.parent = node
        if node.right is not None:
            node.right.parent = node
        check_balance_node = node
    else:
        # case 3: both children
        if kind == "successor":
            assert node.right is not None  # satisfy linter
            successor = search(
                node.right, get_min(node.right)
            )  # could of course be conflated into a single step
            assert successor is not None  # satisfy linter
            # exchange values
            node.value = successor.value
            # replace successor with successor.right
            # why does this work?
            # exchanging values of node and successor is elegant here
            # we then only have to make sure the parent of successor is linked to successor's potential right child
            # Recall
            # - successor must not have a left child as it is the minimum of the right sub-tree of node
            # - successor may or may not have a right child
            # - if successor is the only node in the right sub-tree, we now link node to `None` (just removing)
            # - Ex L9 Slide 32: if successor is some non-leaf node, e.g. 6, then we have set 4 to 6 and now only need to link original 6's parent to 7
            check_balance_node = successor.parent
            remove_or_replace(successor, successor.right)

        else:
            assert node.left is not None  # satisfy linter
            predecessor = search(
                node.left, get_max(node.left)
            )  # could of course be conflated into a single step
            assert predecessor is not None  # satisfy linter
            # exchange values
            node.value = predecessor.value
            # replace predecessor with predecessor.right
            # why does this work?
            # exchanging values of node and predecessor is elegant here
            # we then only have to make sure the parent of predecessor is linked to predecessor's potential right child
            # Recall
            # - predecessor must not have a left child as it is the minimum of the right sub-tree of node
            # - predecessor may or may not have a right child
            # - if predecessor is the only node in the right sub-tree, we now link node to `None` (just removing)
            # - Ex L9 Slide 32: if predecessor is some non-leaf node, e.g. 6, then we have set 4 to 6 and now only need to link original 6's parent to 7
            check_balance_node = predecessor.parent
            remove_or_replace(predecessor, predecessor.left)

    while check_balance_node is not None:
        if is_balanced(check_balance_node):
            check_balance_node = check_balance_node.parent
        else:
            _rebalance(check_balance_node)
            break
    assert is_balanced(check_balance_node)


# Finally, test your implementations with the below template.

# In[13]:


import unittest


class TestAVLTree(unittest.TestCase):
    def test_get_min(self):
        node = Node(12)
        self.assertEqual(get_min(node), 12)
        node.left = Node(6)
        self.assertEqual(get_min(node), 6)
        node.right = Node(19)
        self.assertEqual(get_min(node), 6)
        node.left.left = Node(3)
        self.assertEqual(get_min(node), 3)
        node.left.right = Node(9)
        self.assertEqual(get_min(node), 3)
        node.right.left = Node(14)
        self.assertEqual(get_min(node), 3)
        node.right.right = Node(24)
        self.assertEqual(get_min(node), 3)
        node.left.left.left = Node(2)
        self.assertEqual(get_min(node), 2)
        node.left.left.right = Node(5)
        self.assertEqual(get_min(node), 2)
        node.left.right.left = Node(8)
        self.assertEqual(get_min(node), 2)
        node.left.right.right = Node(11)
        self.assertEqual(get_min(node), 2)
        node.right.left.left = Node(13)
        self.assertEqual(get_min(node), 2)
        node.right.left.right = Node(17)
        self.assertEqual(get_min(node), 2)
        node.right.right.left = Node(21)
        self.assertEqual(get_min(node), 2)
        node.right.right.right = Node(27)
        self.assertEqual(get_min(node), 2)
        self.assertEqual(get_min(node.right), 13)
        self.assertEqual(get_min(node.right.right), 21)

    def test_get_max(self):
        node = Node(12)
        self.assertEqual(get_max(node), 12)
        node.left = Node(6)
        self.assertEqual(get_max(node), 12)
        node.right = Node(19)
        self.assertEqual(get_max(node), 19)
        node.left.left = Node(3)
        self.assertEqual(get_max(node), 19)
        node.left.right = Node(9)
        self.assertEqual(get_max(node), 19)
        node.right.left = Node(14)
        self.assertEqual(get_max(node), 19)
        node.right.right = Node(24)
        self.assertEqual(get_max(node), 24)
        node.left.left.left = Node(2)
        self.assertEqual(get_max(node), 24)
        node.left.left.right = Node(5)
        self.assertEqual(get_max(node), 24)
        node.left.right.left = Node(8)
        self.assertEqual(get_max(node), 24)
        node.left.right.right = Node(11)
        self.assertEqual(get_max(node), 24)
        node.right.left.left = Node(13)
        self.assertEqual(get_max(node), 24)
        node.right.left.right = Node(17)
        self.assertEqual(get_max(node), 24)
        node.right.right.left = Node(21)
        self.assertEqual(get_max(node), 24)
        node.right.right.right = Node(27)
        self.assertEqual(get_max(node), 27)
        self.assertEqual(get_max(node.left), 11)

    def test_is_balanced(self):
        root = Node(1)
        self.assertEqual(is_balanced(root), True)
        root.left = Node(2)
        self.assertEqual(is_balanced(root), True)
        root.right = Node(3)
        self.assertEqual(is_balanced(root), True)
        root.left.left = Node(4)
        self.assertEqual(is_balanced(root), True)
        root.right.left = Node(5)
        self.assertEqual(is_balanced(root), True)
        root.right.left.left = Node(6)
        self.assertEqual(is_balanced(root), False)
        root.left.left.left = Node(7)
        self.assertEqual(is_balanced(root), False)

    #
    def test_is_binary_search_tree(self):
        node = Node(12)
        node.left = Node(6)
        node.right = Node(19)
        node.left.left = Node(3)
        node.left.right = Node(9)
        node.right.left = Node(14)
        node.right.right = Node(24)
        node.left.left.left = Node(2)
        node.left.left.right = Node(5)
        node.left.right.left = Node(8)
        node.left.right.right = Node(11)
        node.right.left.left = Node(13)
        node.right.left.right = Node(17)
        node.right.right.left = Node(21)
        node.right.right.right = Node(27)
        self.assertEqual(True, is_binary_search_tree(node))
        node = Node(12)
        node.left = Node(6)
        node.right = Node(19)
        node.left.left = Node(3)
        node.left.right = Node(9)
        node.right.left = Node(14)
        node.right.right = Node(24)
        node.left.left.left = Node(82)
        node.left.left.right = Node(5)
        node.left.right.left = Node(8)
        node.left.right.right = Node(11)
        node.right.left.left = Node(13)
        node.right.left.right = Node(17)
        node.right.right.left = Node(21)
        node.right.right.right = Node(27)
        self.assertEqual(False, is_binary_search_tree(node))
        self.assertEqual(False, is_binary_search_tree(node.left))
        self.assertEqual(True, is_binary_search_tree(node.right))
        node = Node(12)
        node.left = Node(6)
        node.right = Node(19)
        node.left.left = Node(3)
        node.left.right = Node(9)
        node.right.left = Node(9999)
        node.right.right = Node(24)
        node.left.left.left = Node(2)
        node.left.left.right = Node(5)
        node.left.right.left = Node(8)
        node.left.right.right = Node(11)
        node.right.left.left = Node(13)
        node.right.left.right = Node(17)
        node.right.right.left = Node(21)
        node.right.right.right = Node(27)
        self.assertEqual(True, is_binary_search_tree(node.left))
        self.assertEqual(False, is_binary_search_tree(node.right))

    def test_search(self):
        node = Node(12)
        node.left = Node(12)
        node.left.left = Node(12)
        node.left.right = Node(12)
        node.right = Node(12)
        self.assertIs(search(node, 12), node.right)
        node.right.value = 20
        self.assertIs(search(node, 12), node.left.right)
        node.left.right = Node(14)
        self.assertIs(search(node, 12), node.left.left)
        node.left.left.value = 10
        self.assertIs(search(node, 12), node.left)

    #
    # # As a cautionary note: the tests for insert and deletion are very likely not fully exhaustive.
    def test_insert(self):
        node = Node(9)
        insert(node, 15)
        self.assertEqual(node.right.value, 15)
        insert(node, 20)
        self.assertEqual(node.value, 15)
        self.assertEqual(node.left.value, 9)
        self.assertEqual(node.right.value, 20)
        insert(node, 8)
        self.assertEqual(node.left.left.value, 8)
        insert(node, 7)
        self.assertEqual(node.left.value, 8)
        self.assertEqual(node.left.left.value, 7)
        self.assertEqual(node.left.right.value, 9)
        insert(node, 13)
        self.assertEqual(node.value, 9)
        self.assertEqual(node.right.value, 15)
        self.assertEqual(node.right.left.value, 13)
        self.assertEqual(node.right.right.value, 20)
        insert(node, 10)
        self.assertEqual(node.right.left.left.value, 10)

        node = Node(3)
        insert(node, 5)
        self.assertEqual(node.right.value, 5)
        insert(node, 10)
        self.assertEqual(node.value, 5)
        self.assertEqual(node.left.value, 3)
        self.assertEqual(node.right.value, 10)
        insert(node, 15)
        self.assertEqual(node.right.right.value, 15)
        insert(node, 23)
        self.assertEqual(node.right.value, 15)
        self.assertEqual(node.right.left.value, 10)
        self.assertEqual(node.right.right.value, 23)
        insert(node, 1)
        self.assertEqual(node.left.left.value, 1)
        insert(node, 4)
        self.assertEqual(node.left.right.value, 4)
        insert(node, 100)
        self.assertEqual(node.right.right.right.value, 100)

    def test_delete(self):
        node = Node(5)
        insert(node, 3)
        insert(node, 15)
        insert(node, 1)
        insert(node, 4)
        insert(node, 10)
        insert(node, 23)
        delete(node, 5)
        self.assertEqual(node.value, 10)
        self.assertEqual(node.left.value, 3)
        self.assertEqual(node.left.left.value, 1)
        self.assertEqual(node.left.right.value, 4)
        self.assertEqual(node.right.value, 15)
        delete(node, 4)
        self.assertEqual(node.left.right, None)
        delete(node, 10)
        delete(node, 15)
        self.assertEqual(node.value, 3)
        self.assertEqual(node.left.value, 1)
        self.assertEqual(node.right.value, 23)

    def test_delete_predecessor(self):
        node = Node(9)
        insert(node, 8)
        insert(node, 15)
        insert(node, 7)
        insert(node, 13)
        insert(node, 20)
        insert(node, 10)
        delete(node, 9, kind="predecessor")
        self.assertEqual(node.value, 13)
        self.assertEqual(node.left.value, 8)
        self.assertEqual(node.left.left.value, 7)
        self.assertEqual(node.left.right.value, 10)
        self.assertEqual(node.right.value, 15)
        self.assertEqual(node.right.right.value, 20)
        delete(node, 13, kind="predecessor")
        self.assertEqual(node.value, 10)
        self.assertEqual(node.left.value, 8)
        self.assertEqual(node.left.left.value, 7)
        self.assertEqual(node.right.value, 15)
        self.assertEqual(node.right.right.value, 20)
        #
        node = Node(5)
        insert(node, 3)
        insert(node, 15)
        insert(node, 1)
        insert(node, 4)
        insert(node, 10)
        insert(node, 23)
        delete(node, 5, kind="predecessor")
        self.assertEqual(node.value, 4)
        self.assertEqual(node.left.value, 3)
        self.assertEqual(node.left.left.value, 1)
        self.assertEqual(node.right.value, 15)
        self.assertEqual(node.right.left.value, 10)
        self.assertEqual(node.right.right.value, 23)
        #
        node = Node(5)
        insert(node, 3)
        insert(node, 15)
        insert(node, 1)
        insert(node, 4)
        insert(node, 10)
        insert(node, 23)
        delete(node, 15, kind="predecessor")
        self.assertEqual(node.value, 5)
        self.assertEqual(node.left.value, 3)
        self.assertEqual(node.left.left.value, 1)
        self.assertEqual(node.left.right.value, 4)
        self.assertEqual(node.right.value, 10)
        self.assertEqual(node.right.right.value, 23)
        delete(node, 5, kind="predecessor")
        self.assertEqual(node.value, 4)
        self.assertEqual(node.left.value, 3)
        self.assertEqual(node.left.left.value, 1)
        self.assertEqual(node.right.value, 10)
        self.assertEqual(node.right.right.value, 23)
        delete(node, 3, kind="predecessor")
        delete(node, 1, kind="predecessor")
        self.assertEqual(node.value, 10)
        self.assertEqual(node.left.value, 4)
        self.assertEqual(node.right.value, 23)


unittest.main(argv=[""], verbosity=2, exit=False)
