function string.split(inputstr, sep)
	if sep == nil then
		sep = "%s"
	end
	local t = {}
	for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
		table.insert(t, str)
	end
	return t
end

function love.load(arg)
	-- background
	love.graphics.setBackgroundColor(255, 255, 255, 255)

	-- filesystem
	love.filesystem.setIdentity("problemn", false)

	-- fps
	fps = 60
	fpsth = 1/fps
	fps_time_passed = 0

	-- text
	font_size = 26
	if love.filesystem.getInfo("fonts/UbuntuMono-R.ttf") then
		default_font = love.graphics.newFont("fonts/UbuntuMono-R.ttf", 16)
	else
		default_font = love.graphics.newFont(16)
	end
	love.graphics.setFont(default_font)

	-- render
	do_render = true
	time_since_last_render = 0

	-- enable key repeat so backspace can be held down to trigger love.keypressed multiple times.
	love.keyboard.setKeyRepeat(true)

	-- circle mesh
	local create_circle = function(segments)
		segments = segments or 40
		local vertices = {}

		-- The first vertex is at the origin (0, 0) and will be the center of the circle.
		table.insert(vertices, {0, 0})

		-- Create the vertices at the edge of the circle.
		for i=0, segments do
			local angle = (i / segments) * math.pi * 2

			-- Unit-circle.
			local x = math.cos(angle)
			local y = math.sin(angle)

			table.insert(vertices, {x, y})
		end

		-- The "fan" draw mode is perfect for our circle.
		return love.graphics.newMesh(vertices, "fan")
	end
	circle = create_circle(5)

	-- set variables
	mouse_drag_pos = {x = 0, y = 0}
	is_mouse_pressed = {}
	last_clipboard_text = ""
	input = nil
	decode_status = false
	decode_error = "no input"
	scale = 64
	x_offset = 64
	y_offset = 64
end

function love.run()
	-- ProFi
	-- profi = require('ProFi')
	-- profi:start()

	if love.math then
		love.math.setRandomSeed(os.time())
	end

	if love.load then love.load(arg) end

	-- We don't want the first frame's dt to include time taken by love.load.
	if love.timer then love.timer.step() end

	local dt = 0

	-- Main loop time.
	while true do
		-- Process events.
		if love.event then
			love.event.pump()
			for name, a,b,c,d,e,f in love.event.poll() do
				if name == "quit" then
					if not love.quit or not love.quit() then
						-- ProFi
						-- profi:stop()
						-- profi:writeReport('report.txt')

						return a
					end
				end
				love.handlers[name](a,b,c,d,e,f)
			end
		end

		-- Update dt, as we'll be passing it to update
		if love.timer then
			love.timer.step()
			dt = love.timer.getDelta()
			fps_time_passed = fps_time_passed + dt
		end

		-- Call update and draw
		if love.update then love.update(dt) end -- will pass 0 if love.timer is disabled

		if love.graphics and love.graphics.isActive() then
			if do_render then
				do_render = nil
				time_since_last_render = 0
				love.graphics.clear(love.graphics.getBackgroundColor())
				love.graphics.origin()
				if love.draw then love.draw() end
				fps_time_passed = fps_time_passed - fpsth
				love.graphics.present()
			end
		end

		if love.timer then love.timer.sleep(.001) end
	end
end

function decode_input_text(text)
	local r = {}

	local pipe_splits = string.split(text, '|')

	-- problem input
	do
		local input_text = pipe_splits[1]
		local lines = string.split(input_text, '\n')

		local line_index = 1
		local number_of_inner_points = tonumber(lines[line_index])
		local inner_points = {}
		for i = line_index + 1, line_index + number_of_inner_points do
			local numbers = string.split(lines[i], ' ')
			local x, y = tonumber(numbers[1]), tonumber(numbers[2])
			if x ~= nil and y ~= nil then
				table.insert(inner_points, {x = tonumber(numbers[1]), y = tonumber(numbers[2])})
			end
		end
		line_index = line_index + 1 + number_of_inner_points

		local number_of_outer_points = tonumber(lines[line_index])
		local outer_points = {}
		for i = line_index + 1, line_index + number_of_outer_points do
			local numbers = string.split(lines[i], ' ')
			local x, y = tonumber(numbers[1]), tonumber(numbers[2])
			if x ~= nil and y ~= nil then
				table.insert(outer_points, {x = x, y = y})
			end
		end
		line_index = line_index + 1 + number_of_outer_points

		r.problem = {
			inner_points = inner_points,
			outer_points = outer_points
		}
	end

	-- solution input
	if #pipe_splits >= 2 then
		local input_text = pipe_splits[2]
		local text_lines = string.split(input_text, '\n')

		local line_index = 1
		local lines = {}
		while text_lines[line_index] ~= nil do
			local numbers = string.split(text_lines[line_index], ' ')
			if #numbers == 2 then
				local x, y = tonumber(numbers[1]), tonumber(numbers[2])
				if x ~= nil and y ~= nil then
					table.insert(lines, {type = "point", x = x, y = y})
				end
			else
				local x1, y1, x2, y2 = tonumber(numbers[1]), tonumber(numbers[2]), tonumber(numbers[3]), tonumber(numbers[4])
				if x1 ~= nil and y1 ~= nil and x2 ~= nil and y2 ~= nil then
					table.insert(lines, {type = "line", x1 = x1, y1 = y1, x2 = x2, y2 = y2})
				end
			end
			line_index = line_index + 1
		end

		r.solution = {
			lines = lines
		}
	end

	return r
end

function check_clipboard_text()
	local current_clipboard_text = love.system.getClipboardText();

	if current_clipboard_text == last_clipboard_text then return end
	last_clipboard_text = current_clipboard_text

	decode_status, decode_error = xpcall(function()
		input = decode_input_text(current_clipboard_text)
	end, debug.traceback)

	do_render = true
end

function love.update(dt)
	check_clipboard_text()
	time_since_last_render = time_since_last_render + dt
	if time_since_last_render > 5.0 then
		do_render = true
	end
end

function display_to_world_x(display_x)
	return (display_x - x_offset) / scale
end

function display_to_world_y(display_y)
	return (display_y - y_offset) / scale
end

function world_to_display_x(world_x)
	return x_offset + world_x * scale
end

function world_to_display_y(world_y)
	return y_offset + world_y * scale
end

function draw_background_dots()
	if scale < 15 then
		love.graphics.setLineStyle("rough")
		for xpos = math.floor(display_to_world_x(0)), math.ceil(display_to_world_x(love.graphics.getWidth())), 1 do
			if xpos == 0 then
				love.graphics.setColor(1.0, 1.0, 1.0, 0.2)
				love.graphics.setLineWidth(2)
				local display_x = math.floor(world_to_display_x(xpos))
				love.graphics.line(display_x, 0, display_x, love.graphics.getHeight())
			else
				love.graphics.setColor(1.0, 1.0, 1.0, 0.1)
				love.graphics.setLineWidth(1)
				local display_x = math.floor(world_to_display_x(xpos))
				love.graphics.line(display_x, 0, display_x, love.graphics.getHeight())
			end
		end

		for ypos = math.floor(display_to_world_y(0)), math.ceil(display_to_world_y(love.graphics.getHeight())), 1 do
			if ypos == 0 then
				love.graphics.setColor(1.0, 1.0, 1.0, 0.2)
				love.graphics.setLineWidth(2)
				local display_y = math.floor(world_to_display_y(ypos))
				love.graphics.line(0, display_y, love.graphics.getWidth(), display_y)
			else
				love.graphics.setColor(1.0, 1.0, 1.0, 0.1)
				love.graphics.setLineWidth(1)
				local display_y = math.floor(world_to_display_y(ypos))
				love.graphics.line(0, display_y, love.graphics.getWidth(), display_y)
			end
		end
		return
	end

	for xpos = math.floor(display_to_world_x(0)), math.ceil(display_to_world_x(love.graphics.getWidth())), 1 do
		for ypos = math.floor(display_to_world_y(0)), math.ceil(display_to_world_y(love.graphics.getHeight())), 1 do
			if xpos == 0 or ypos == 0 then
				love.graphics.setColor(1.0, 1.0, 1.0, 0.4)
				love.graphics.draw(circle, math.floor(world_to_display_x(xpos)), math.floor(world_to_display_y(ypos)), 0, 3, 3)
			else
				love.graphics.setColor(1.0, 1.0, 1.0, 0.2)
				love.graphics.draw(circle, math.floor(world_to_display_x(xpos)), math.floor(world_to_display_y(ypos)), 0, 2, 2)
			end
		end
	end
end

function draw_dashed_line(x1, y1, x2, y2)
	local x, y = x2 - x1, y2 - y1
	local len = math.sqrt(x^2 + y^2)
	local stepx, stepy = x / len, y / len
	x = x1
	y = y1

	for i = 1, len do
		if i % 8 == 0 then
			love.graphics.circle("fill", math.floor(x), math.floor(y), 1)
		end
		x = x + stepx
		y = y + stepy
	end
end

function draw_polygons()
	if input.problem == nil then return end

	local draw_path = function(points, color, is_dashed)
		local index = 1
		local point = points[index]

		local first_point = point
		local last_point

		local draw_line = draw_dashed_line
		if not is_dashed then
			draw_line = love.graphics.line
			love.graphics.setLineStyle("rough")
			love.graphics.setLineWidth(2)
		end

		while point ~= nil do
			index = index + 1
			local new_point = points[index]
			if new_point == nil then
				point = nil
				break
			end

			last_point = new_point

			love.graphics.setColor(color)
			draw_line(
				world_to_display_x(point.x),
				world_to_display_y(-point.y),
				world_to_display_x(new_point.x),
				world_to_display_y(-new_point.y)
			)

			point = new_point
		end

		if last_point ~= nil then
			draw_line(
				world_to_display_x(last_point.x),
				world_to_display_y(-last_point.y),
				world_to_display_x(first_point.x),
				world_to_display_y(-first_point.y)
			)
		end
	end

	local draw_lines = function(lines, color0, color1, is_dashed)
		local index = 1
		local line = lines[index]

		local draw_line = draw_dashed_line
		if not is_dashed then
			draw_line = love.graphics.line
			love.graphics.setLineStyle("rough")
		end

		while line ~= nil do
			love.graphics.setColor(color0)
			if line.type == "line" then
				draw_line(
					world_to_display_x(line.x1),
					world_to_display_y(-line.y1),
					world_to_display_x(line.x2),
					world_to_display_y(-line.y2)
				)
			elseif line.type == "point" then
				love.graphics.setColor(color1)
				love.graphics.circle("line", world_to_display_x(line.x), world_to_display_y(-line.y), 5)
			end

			index = index + 1
			line = lines[index]
		end
	end

	local path_dim = 1.0
	if input.solution then
		path_dim = 0.5
	end
	draw_path(input.problem.outer_points, {0.3 * path_dim, 0.68 * path_dim, 1.0 * path_dim})
	draw_path(input.problem.inner_points, {255 / 255 * path_dim, 136 / 255 * path_dim, 71 / 255 * path_dim})
	if input.solution then
		draw_lines(input.solution.lines, {60 / 255, 224 / 255, 101 / 255}, {255 / 255, 140 / 255, 30 / 255}, true)
	end
end

function love.draw()
	love.graphics.clear(0.18, 0.18, 0.22)

	if not decode_status or input == nil then
		local text = love.graphics.newText(
			default_font,
			{
				{1.0, 0.38, 0.0}, decode_error .. "\n\n\n",
				{1.0, 1.0, 1.0}, "Clipboard text:\n" .. last_clipboard_text
			}
		)
		love.graphics.setColor(1.0, 1.0, 1.0)
		love.graphics.draw(text, 16, 16)
	else
		local status, error = xpcall(function()
			draw_background_dots()

			draw_polygons()

			love.graphics.setColor(1.0, 1.0, 1.0);
			love.graphics.print(
				string.format("scale: %.0f (ctrl + mousewheel to change)\noffset: %.0f, %.0f ((shift +) mousewheel or MB2 to change)", scale, x_offset, y_offset),
				8, 8
			)
		end, debug.traceback)

		if not status then
			local text = love.graphics.newText(
				default_font,
				{
					{1.0, 0.38, 1.0}, error .. "\n\n\n",
					{1.0, 1.0, 1.0}, "Clipboard text:\n" .. last_clipboard_text
				}
			)
			love.graphics.setColor(1.0, 1.0, 1.0)
			love.graphics.draw(text, 16, 16)
		end
	end
end

function love.keypressed(key)
	do_render = true


	if key == "+" then
		scale = scale + scale * 0.2

    elseif key == "-" then
		scale = math.max(1, scale - scale * 0.2)
	end
end

function love.textinput(t)
    -- input = input .. t
	do_render = true

	
end

function love.mousemoved(x, y, dx, dy, istouch)
	if is_mouse_pressed[2] then
		do_render = true
		x_offset = x - mouse_drag_pos.x
		y_offset = y - mouse_drag_pos.y
	end
end

function love.mousereleased(x, y, button, istouch, presses)
	
end

function love.mousepressed(x, y, button, istouch, presses)
	do_render = true

	if button == 2 then
		mouse_drag_pos = {x = x - x_offset, y = y - y_offset}
	end

	is_mouse_pressed[button] = true
end

function love.mousereleased(x, y, button, istouch, presses)
	do_render = true
	
	is_mouse_pressed[button] = false
end

function love.wheelmoved(x, y)
	do_render = true

	if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
		if y > 0 then
			scale = scale + scale * 0.1
		end
		if y < 0 then
			scale = math.max(1, scale - scale * 0.1)
		end

	elseif love.keyboard.isDown("lshift") or love.keyboard.isDown("rshift") then
		if y > 0 then
			x_offset = x_offset + 1024 / scale
		end
		if y < 0 then
			x_offset = x_offset - 1024 / scale
		end

	else
		if y > 0 then
			y_offset = y_offset + 1024 / scale
		end
		if y < 0 then
			y_offset = y_offset - 1024 / scale
		end

		if x > 0 then
			x_offset = x_offset - 1024 / scale
		end
		if x < 0 then
			x_offset = x_offset + 1024 / scale
		end
	end
end

function love.resize(w, h)
	do_render = true
end

function love.quit()

end
