function string.split(inputstr, sep)
	if sep == nil then
		sep = "%s"
	end
	local t = {}
	for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
		table.insert(t, str)
	end
	return t
end

function love.load(arg)
	-- background
	love.graphics.setBackgroundColor(255, 255, 255, 255)

	-- filesystem
	love.filesystem.setIdentity("problemn problem maker", false)

	-- fps
	fps = 60
	fpsth = 1/fps
	fps_time_passed = 0

	-- text
	font_size = 26
	if love.filesystem.getInfo("fonts/UbuntuMono-R.ttf") then
		default_font = love.graphics.newFont("fonts/UbuntuMono-R.ttf", 16)
	else
		default_font = love.graphics.newFont(16)
	end
	love.graphics.setFont(default_font)

	-- render
	do_render = true
	time_since_last_render = 0

	-- enable key repeat so backspace can be held down to trigger love.keypressed multiple times.
	love.keyboard.setKeyRepeat(true)

	-- circle mesh
	local create_circle = function(segments)
		segments = segments or 40
		local vertices = {}

		-- The first vertex is at the origin (0, 0) and will be the center of the circle.
		table.insert(vertices, {0, 0})

		-- Create the vertices at the edge of the circle.
		for i=0, segments do
			local angle = (i / segments) * math.pi * 2

			-- Unit-circle.
			local x = math.cos(angle)
			local y = math.sin(angle)

			table.insert(vertices, {x, y})
		end

		-- The "fan" draw mode is perfect for our circle.
		return love.graphics.newMesh(vertices, "fan")
	end
	circle = create_circle(5)

	-- set variables
	scale = 64
	x_offset = 256
	y_offset = 256
	mouse_drag_pos = {x = 0, y = 0}
	is_mouse_pressed = {}
	mouse_hover_world_pos_int = {x = 0, y = 0}
	polys = {
		inner_poly = {
			{-1, -1},
			{-1,  1},
			{ 1,  1},
			{ 1, -1},
		},
		outer_poly = {
			{-2, -2},
			{-2,  2},
			{ 2,  2},
			{ 2, -2},
		}
	}
	undo_history = {}
	redo_history = {}
	dragged_node = nil
end

function love.run()
	-- ProFi
	-- profi = require('ProFi')
	-- profi:start()

	if love.math then
		love.math.setRandomSeed(os.time())
	end

	if love.load then love.load(arg) end

	-- We don't want the first frame's dt to include time taken by love.load.
	if love.timer then love.timer.step() end

	local dt = 0

	-- Main loop time.
	while true do
		-- Process events.
		if love.event then
			love.event.pump()
			for name, a,b,c,d,e,f in love.event.poll() do
				if name == "quit" then
					if not love.quit or not love.quit() then
						-- ProFi
						-- profi:stop()
						-- profi:writeReport('report.txt')

						return a
					end
				end
				love.handlers[name](a,b,c,d,e,f)
			end
		end

		-- Update dt, as we'll be passing it to update
		if love.timer then
			love.timer.step()
			dt = love.timer.getDelta()
			fps_time_passed = fps_time_passed + dt
		end

		-- Call update and draw
		if love.update then love.update(dt) end -- will pass 0 if love.timer is disabled

		if love.graphics and love.graphics.isActive() then
			if do_render then
				do_render = false
				time_since_last_render = 0
				love.graphics.clear(love.graphics.getBackgroundColor())
				love.graphics.origin()
				if love.draw then love.draw() end
				fps_time_passed = fps_time_passed - fpsth
				love.graphics.present()
			end
		end

		if love.timer then love.timer.sleep(.001) end
	end
end

function love.update(dt)
	time_since_last_render = time_since_last_render + dt
	if time_since_last_render > 5.0 then
		do_render = true
	end
end

function draw_dashed_line(x1, y1, x2, y2)
	local x, y = x2 - x1, y2 - y1
	local len = math.sqrt(x^2 + y^2)
	local stepx, stepy = x / len, y / len
	x = x1
	y = y1

	for i = 1, len do
		if i % 8 == 0 then
			love.graphics.circle("fill", math.floor(x), math.floor(y), 1)
		end
		x = x + stepx
		y = y + stepy
	end
end

function display_to_world_x(display_x)
	return (display_x - x_offset) / scale
end

function display_to_world_y(display_y)
	return (display_y - y_offset) / scale
end

function world_to_display_x(world_x)
	return x_offset + world_x * scale
end

function world_to_display_y(world_y)
	return y_offset + world_y * scale
end

function draw_background_dots()
	if scale < 15 then
		love.graphics.setLineStyle("rough")
		for xpos = math.floor(display_to_world_x(0)), math.ceil(display_to_world_x(love.graphics.getWidth())), 1 do
			if xpos == 0 then
				love.graphics.setColor(1.0, 1.0, 1.0, 0.2)
				love.graphics.setLineWidth(2)
				local display_x = math.floor(world_to_display_x(xpos))
				love.graphics.line(display_x, 0, display_x, love.graphics.getHeight())
			else
				love.graphics.setColor(1.0, 1.0, 1.0, 0.1)
				love.graphics.setLineWidth(1)
				local display_x = math.floor(world_to_display_x(xpos))
				love.graphics.line(display_x, 0, display_x, love.graphics.getHeight())
			end
		end

		for ypos = math.floor(display_to_world_y(0)), math.ceil(display_to_world_y(love.graphics.getHeight())), 1 do
			if ypos == 0 then
				love.graphics.setColor(1.0, 1.0, 1.0, 0.2)
				love.graphics.setLineWidth(2)
				local display_y = math.floor(world_to_display_y(ypos))
				love.graphics.line(0, display_y, love.graphics.getWidth(), display_y)
			else
				love.graphics.setColor(1.0, 1.0, 1.0, 0.1)
				love.graphics.setLineWidth(1)
				local display_y = math.floor(world_to_display_y(ypos))
				love.graphics.line(0, display_y, love.graphics.getWidth(), display_y)
			end
		end
		return
	end

	for xpos = math.floor(display_to_world_x(0)), math.ceil(display_to_world_x(love.graphics.getWidth())), 1 do
		for ypos = math.floor(display_to_world_y(0)), math.ceil(display_to_world_y(love.graphics.getHeight())), 1 do
			if xpos == 0 or ypos == 0 then
				love.graphics.setColor(1.0, 1.0, 1.0, 0.4)
				love.graphics.draw(circle, math.floor(world_to_display_x(xpos)), math.floor(world_to_display_y(ypos)), 0, 3, 3)
			else
				love.graphics.setColor(1.0, 1.0, 1.0, 0.2)
				love.graphics.draw(circle, math.floor(world_to_display_x(xpos)), math.floor(world_to_display_y(ypos)), 0, 2, 2)
			end
		end
	end
end

function draw_poly(poly, color)
	love.graphics.setLineStyle("rough")
	love.graphics.setLineWidth(2.0)
	love.graphics.setColor(color)
	for i = 1, #poly - 1 do
		local pos1x, pos1y = world_to_display_x(poly[i][1]), world_to_display_y(poly[i][2])
		local pos2x, pos2y = world_to_display_x(poly[i + 1][1]), world_to_display_y(poly[i + 1][2])
		love.graphics.line(
			pos1x, pos1y,
			pos2x, pos2y
		)
	end
	love.graphics.line(
		world_to_display_x(poly[1][1]), world_to_display_y(poly[1][2]),
		world_to_display_x(poly[#poly][1]), world_to_display_y(poly[#poly][2])
	)
end

function get_polygon_node_from_poly(x, y, poly)
	for i = 1, #poly do
		if poly[i][1] == x and poly[i][2] == y then
			return i
		end
	end
	return nil
end

function get_polygon_node(x, y)
	for k, v in pairs(polys) do
		local index = get_polygon_node_from_poly(x, y, v)
		if index ~= nil then
			return {poly = k, index = index}
		end
	end
	return nil
end

function draw_world_cursor(color1, color2)
	local x, y = mouse_hover_world_pos_int.x, mouse_hover_world_pos_int.y
	love.graphics.setColor(1.0, 1.0, 1.0, 0.6)
	local size = 4
	local poly_index_text = ""
	local inner_poly_index = get_polygon_node_from_poly(x, y, polys.inner_poly)
	if inner_poly_index ~= nil then
		love.graphics.setColor(color1)
		size = 6
		poly_index_text = poly_index_text .. inner_poly_index
	end
	local outer_poly_index = get_polygon_node_from_poly(x, y, polys.outer_poly)
	if outer_poly_index ~= nil then
		love.graphics.setColor(color2)
		size = 6
		poly_index_text = poly_index_text .. outer_poly_index
	end
	love.graphics.circle(
		"fill",
		world_to_display_x(x),
		world_to_display_y(y),
		size
	)
	love.graphics.print(poly_index_text, 8, love.graphics.getHeight() - default_font:getHeight() - 8)
end

function love.draw()
	love.graphics.clear(0.133333, 0.164706, 0.219608)

	draw_background_dots()

	local color1, color2 = {255 / 255, 136 / 255, 71 / 255}, {57 / 255, 166 / 255, 255 / 255}
	draw_poly(polys.inner_poly, color1)
	draw_poly(polys.outer_poly, color2)

	draw_world_cursor(color1, color2)

	love.graphics.setColor(1.0, 1.0, 1.0);
	love.graphics.print(
		string.format("scale: %.0f (ctrl + mousewheel to change)\noffset: %.0f, %.0f ((shift +) mousewheel or MB2 to change)\n\ndrag node: MB1\nclone node: ctrl + MB1\ndelete node: del (while dragging node)\ncopy: ctrl + c\npaste: ctrl + v\nnew file: ctrl + n\ninvert polygon order: ctrl + 1 or ctrl + 2", scale, x_offset, y_offset),
		8, 8
	)
end

function deep_copy(some_table)
	if type(some_table) ~= "table" then
		return some_table
	end

	local r = {}
	for k, v in pairs(some_table) do
		r[deep_copy(k)] = deep_copy(v)
	end
	return r
end

function update_undo_history()
	undo_history[#undo_history + 1] = deep_copy(polys)
	if #undo_history > 128 then
		table.remove(undo_history, 1)
	end
	redo_history = {}
end

function undo()
	if #undo_history == 0 then
		return
	end

	redo_history[#redo_history + 1] = polys
	polys = undo_history[#undo_history]
	undo_history[#undo_history] = nil
end

function redo()
	if #redo_history == 0 then
		return
	end

	undo_history[#undo_history + 1] = polys
	polys = redo_history[#redo_history]
	redo_history[#redo_history] = nil
end

function decode_input_text(text)
	local pipe_splits = string.split(text, '|')

	-- problem input
	do
		local input_text = pipe_splits[1]
		local lines = string.split(input_text, '\n')

		local line_index = 1
		local number_of_inner_points = tonumber(lines[line_index])
		local inner_points = {}
		for i = line_index + 1, line_index + number_of_inner_points do
			local numbers = string.split(lines[i], ' ')
			local x, y = tonumber(numbers[1]), tonumber(numbers[2])
			if x ~= nil and y ~= nil then
				table.insert(inner_points, {x, -y})
			end
		end
		line_index = line_index + 1 + number_of_inner_points

		local number_of_outer_points = tonumber(lines[line_index])
		local outer_points = {}
		for i = line_index + 1, line_index + number_of_outer_points do
			local numbers = string.split(lines[i], ' ')
			local x, y = tonumber(numbers[1]), tonumber(numbers[2])
			if x ~= nil and y ~= nil then
				table.insert(outer_points, {x, -y})
			end
		end
		line_index = line_index + 1 + number_of_outer_points

		update_undo_history()

		polys.inner_poly = inner_points
		polys.outer_poly = outer_points
	end
end

function generate_input_text()
	local r = ""
	local number_of_inner_points = #polys.inner_poly
	r = r .. number_of_inner_points .. '\n'
	for i = 1, #polys.inner_poly do
		r = r .. string.format("%d %d\n", polys.inner_poly[i][1], -polys.inner_poly[i][2])
	end
	local number_of_outer_points = #polys.outer_poly
	r = r .. number_of_outer_points .. '\n'
	for i = 1, #polys.outer_poly do
		r = r .. string.format("%d %d\n", polys.outer_poly[i][1], -polys.outer_poly[i][2])
	end
	return r
end

function invert_poly(poly_name)
	local inverted = {}
	for i = #polys[poly_name], 1, -1 do
		table.insert(inverted, polys[poly_name][i])
	end
	polys[poly_name] = inverted
end

function love.keypressed(key)
	do_render = true


	if key == "+" then
		scale = scale + scale * 0.2

    elseif key == "-" then
		scale = math.max(1, scale - scale * 0.2)

	elseif key == "v" then
		if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
			local status, error = xpcall(function()
				decode_input_text(love.system.getClipboardText())
			end, debug.traceback)
		end

	elseif key == "c" then
		if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
			love.system.setClipboardText(generate_input_text())
		end

	elseif key == "n" then
		if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
			update_undo_history()
			polys = {
				inner_poly = {
					{-1, -1},
					{-1,  1},
					{ 1,  1},
					{ 1, -1},
				},
				outer_poly = {
					{-2, -2},
					{-2,  2},
					{ 2,  2},
					{ 2, -2},
				}
			}
			dragged_node = nil
		end

	elseif key == "z" then
		if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
			if love.keyboard.isDown("lshift") or love.keyboard.isDown("rshift") then
				redo()
			else
				undo()
			end
		end

	elseif key == "1" then
		if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
			invert_poly("inner_poly")
		end

	elseif key == "2" then
		if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
			invert_poly("outer_poly")
		end

	elseif key == "escape" then
		if dragged_node ~= nil then
			if dragged_node.action == "move" then
				polys[dragged_node.poly][dragged_node.index] = dragged_node.original_position
			elseif dragged_node.action == "clone" then
				table.remove(polys[dragged_node.poly], dragged_node.index)
			end
			dragged_node = nil
		end

	elseif key == "delete" then
		if dragged_node ~= nil then
			if (#polys[dragged_node.poly] > 3) then
				table.remove(polys[dragged_node.poly], dragged_node.index)
				dragged_node = nil
			end
		end

	end
end

function love.textinput(t)
    -- input = input .. t
	do_render = true

end

function love.mousemoved(x, y, dx, dy, istouch)
	if is_mouse_pressed[2] then
		do_render = true
		x_offset = x - mouse_drag_pos.x
		y_offset = y - mouse_drag_pos.y
	end

	if dragged_node ~= nil then
		do_render = true
		polys[dragged_node.poly][dragged_node.index] = {display_to_world_x(x), display_to_world_y(y)}
	end

	local hover_int_x = math.floor(display_to_world_x(x + scale * 0.5))
	local hover_int_y = math.floor(display_to_world_y(y + scale * 0.5))
	if (hover_int_x ~= mouse_hover_world_pos_int.x or hover_int_y ~= mouse_hover_world_pos_int.y) then
		do_render = true
		mouse_hover_world_pos_int = {x = hover_int_x, y = hover_int_y}
	end
end

function love.mousepressed(x, y, button, istouch, presses)
	do_render = true

	if button == 2 then
		mouse_drag_pos = {x = x - x_offset, y = y - y_offset}
	end

	if button == 1 then
		if dragged_node == nil then
			dragged_node = get_polygon_node(mouse_hover_world_pos_int.x, mouse_hover_world_pos_int.y)
			if dragged_node ~= nil then
				update_undo_history()
				dragged_node.action = "move"
				dragged_node.original_position = {mouse_hover_world_pos_int.x, mouse_hover_world_pos_int.y}
				if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
					table.insert(polys[dragged_node.poly], dragged_node.index + 1, {math.floor(display_to_world_x(x + scale * 0.5)), math.floor(display_to_world_y(y + scale * 0.5))})
					dragged_node.action = "clone"
				end
			end
		else
			if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
				table.insert(polys[dragged_node.poly], dragged_node.index + 1, {math.floor(display_to_world_x(x + scale * 0.5)), math.floor(display_to_world_y(y + scale * 0.5))})
				dragged_node.action = "clone"
			else
				polys[dragged_node.poly][dragged_node.index] = {math.floor(display_to_world_x(x + scale * 0.5)), math.floor(display_to_world_y(y + scale * 0.5))}
				dragged_node = nil
			end
		end
	end

	is_mouse_pressed[button] = true
end

function love.mousereleased(x, y, button, istouch, presses)
	do_render = true
	
	is_mouse_pressed[button] = false
end

function love.wheelmoved(x, y)
	do_render = true

	if love.keyboard.isDown("lctrl") or love.keyboard.isDown("rctrl") then
		if y > 0 then
			scale = scale + scale * 0.1
		end
		if y < 0 then
			scale = math.max(1, scale - scale * 0.1)
		end

	elseif love.keyboard.isDown("lshift") or love.keyboard.isDown("rshift") then
		if y > 0 then
			x_offset = x_offset + 1024 / scale
		end
		if y < 0 then
			x_offset = x_offset - 1024 / scale
		end

	else
		if y > 0 then
			y_offset = y_offset + 1024 / scale
		end
		if y < 0 then
			y_offset = y_offset - 1024 / scale
		end

		if x > 0 then
			x_offset = x_offset - 1024 / scale
		end
		if x < 0 then
			x_offset = x_offset + 1024 / scale
		end
	end
end

function love.resize(w, h)
	do_render = true
end

function love.quit()

end
